<?php
/**
 * User Is Not Added List
 *
 * @package     AutomatorWP\Integrations\FluentCRM\Filters\User_Is_Not_Added_List
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_FluentCRM_User_Is_Not_Added_List_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'fluentcrm';
    public $filter = 'fluentcrm_user_is_not_added_list';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User is not in list', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>is not in list</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. %2$s: Tag. */
            'edit_label'        => sprintf( __( '%1$s', 'automatorwp-pro' ), '{list}' ),
            /* translators: %1$s: Condition. %2$s: Tag. */
            'log_label'         => sprintf( __( '%1$s', 'automatorwp-pro' ), '{list}'),
            'options'           => array(
                'list' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'list',
                    'option_default'    => __( 'Select a list', 'automatorwp-pro' ),
                    'name'              => __( 'List:', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'List ID', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_fluentcrm_get_lists',
                    'options_cb'        => 'automatorwp_fluentcrm_options_cb_list',
                    'default'           => ''
                ) ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        global $wpdb;
        
        // Shorthand
        $list_id = absint( $filter_options['list'] );

        // Bail if empty list to assign
        if( empty( $list_id ) ) {
            return;
        }

        $list = $wpdb->get_row( $wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}fc_lists WHERE id = %s",
            $list_id
        ) );

        // Bail if list not exists
        if( ! $list ) {
            $this->result = __( 'Filter not passed. List does not exist.', 'automatorwp-pro' );
            return false;
        }

        $subscriber = automatorwp_fluentcrm_get_subscriber( $user_id );

        // Bail if subscriber not exists
        if( ! $subscriber ) {
            $this->result = __( 'Filter not passed. User is not a subscriber.', 'automatorwp-pro' );
            return false;
        }

        // Don't deserve if user does not have the list
        if( $subscriber->hasAnyListId( array( $list_id ) ) ) {
            $this->result = sprintf( __( 'Filter not passed. User does not meet the condition is not in list "%1$s".', 'automatorwp-pro' ),
                $list->title
            );
            return false;
        }

        $this->result = sprintf( __( 'Filter passed. User is not in list "%1$s".', 'automatorwp-pro' ),
            $list->title
        );

        return $deserves_filter;

    }

}

new AutomatorWP_FluentCRM_User_Is_Not_Added_List_Filter();